#define _POSIX_SOURCE

#include <sys/select.h>
#include <prototypes/inet_proto.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <c_utilities.h>

#define bzero(s, len)             memset((char *)(s), 0, len)

int errno;

ext_shared short port_no;                    /* port to listen on */
ext_shared int   ka_timer;                   /* keep alive timer */
ext_shared short delay;                      /* delay between echos in seconds */
ext_shared short lingertime = 0;             /* linger time in seconds */

ext_shared char output_path [256];           /* something to hold path of PM */

void s$get_task_id();

open_listening_socket (int * socks0a, int ka_timer_a, int lingertime_a)
{
 struct sockaddr_in serv_addr;     /* used to bind to IP address and port */
 int reuse = 1;                    /* flag to control address reuse */
 int nodelay = 1;                  /* flag to control no delay setting */
 int fcntl_flags;                  /* holds flags for fcntl function
                                      used to set non-blocking mode */
 struct linger  linger1;           /* linger structure */

/* create a stream socket */
 if ((*socks0a = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
    perror ("tasking_echo: can't open stream socket");
    exit (errno);
    }

/* set a bunch of sock options on what will become the listening socket.
   all these options will be inherited by the sockets created when a
   connection comes in and an accept is done */

/* Set SO_REUSEADDR so that I do not have to wait for an old socket in 
   TIMEWAIT state to be cleaned up before starting another server */

 if (setsockopt (*socks0a, SOL_SOCKET, SO_REUSEADDR, (char *) &reuse, sizeof (reuse)) < 0)
   {
   perror ("tasking_echo: Error setting reuseaddr");
   exit (errno);
   }

/* set the linger timer if its greater than -1. */

 if (lingertime_a > -1)
    {
    linger1.l_onoff = 1;
    linger1.l_linger = lingertime_a;
    if (setsockopt (*socks0a, SOL_SOCKET, SO_LINGER, (char *) &linger1, sizeof (linger1)) < 0)
       {
       perror ("tasking_echo: Error setting linger");
       exit (errno);
       }
    }

/* turn keep alive on so any connections that are left hanging can be cleaned
   up eventually. */

 if (ka_timer_a != 0)
    {
    linger1.l_onoff = 1;
    linger1.l_linger = ka_timer_a;
    if (setsockopt (*socks0a, SOL_SOCKET, SO_KEEPALIVE, (char *) &linger1, sizeof (linger1)) < 0)
      {
      perror ("tasking_echo: Error setting keepalive");
      exit (errno);
      }
    }

/* turn on no delay. This turns off the nagel algorithm so that we can send
   multiple small (less than MSS) packets without waiting for an ACK. */

 if (setsockopt (*socks0a, IPPROTO_TCP, TCP_NODELAY, (char *) &nodelay, sizeof (nodelay)) < 0)
    {
    perror ("tasking_echo: Error setting nodelay");
    exit (errno);
    }

/* build a sockaddr structure holding the address we will bind to. The IP
   address is INADDR_ANY meaning we will listen on all active IP addresses */

 bzero ( (char *) &serv_addr, sizeof (serv_addr));
 serv_addr.sin_family        = AF_INET;
 serv_addr.sin_addr.s_addr   = htonl (INADDR_ANY);
 serv_addr.sin_port          = htons (port_no);

/* now bind to the address and port */

 if (bind (*socks0a, (struct sockaddr *) &serv_addr, sizeof (serv_addr)) < 0)
    {
    perror ("tasking_echo: can't bind local address");
    exit (errno);
    }

/* set non-blocking mode on the socket. At the moment doing a get and ORing
   in the O_NDELAY value is probably overkill since the only flag supported 
   at this time is O_NDELAY. But things may change in the future so (as was
   explained to be in painful detail with graphic images of bits crashing
   togther) its best to do things right. */

 fcntl_flags = fcntl(*socks0a, F_GETFL, 0);
 if (fcntl_flags < 0)
    {
    perror ("tasking_echo: could not get the current fcntl flags for listening socket");
    exit (errno);
    }

 fcntl_flags = fcntl_flags | O_NDELAY;

 if (fcntl(*socks0a, F_SETFL, fcntl_flags) < 0)
    {
    perror ("tasking_echo: could not set non blocking IO for listening socket");
    exit (errno);
    }

/* listen for connections with a backlog of 1 */

 listen (*socks0a, 1);
}


echo_task ()
{
 struct sockaddr_in cli_addr;      /* holds address of remote from accept */
 int    clilen;                    /* holds length of cli_addr structure) */
 int    nfds;                      /* max sock FD value, used in select */
 fd_set fdsetR;                    /* set of FDs to be tested by select */

 struct timeval timeout;           /* holds select timeout value */
 int fcntl_flags;                  /* holds flags for fcntl function
                                      used to set non-blocking mode */

#define BUFFERLEN 100              /* how many bytes can we process at once */
 int socks0, socks1;               /* socks0 holds the listening socket
                                      socks1 holds the accepted socket */
 char peer [16];                   /* array holding string containing IP
                                      address of remote peer for socks1 */
 char echostring [BUFFERLEN];      /* array holding characters to be
                                           echoed */
 int echostringlen;                /* holds number of character in
                                      echostring array */
 int oldechostringlen;             /* holds previous echostring length */

 short socksSelected;              /* number of sockets that select indicates
                                      are ready */
 int recvBytes, sendBytes;         /* number of bytes received and sent */
 char msg [103];                   /* temporary space for a string */
 char *peer_x;                     /* temporary space to hold the string
                                      containing peer;s IP address */
 short i, j, x, y;                 /* loop counters and temporaries */

 int savedErrno;
 FILE * outFile;
 short int myTaskID;               /* name says it all */

/* get my task ID for messages */

 s$get_task_id (&myTaskID);

/* load the timeout struction based on the input delay value. This will be used
   by select. */

 timeout.tv_sec = delay;
 timeout.tv_usec = 0;

 open_listening_socket (&socks0, ka_timer, lingertime);

/* begining of main loop through the code, now things get complicated */

socks1 = 0;

again:

/* reset the file descriptor array for select. This has to be done every time
   you go through select since select will modify this array

   since we are only handing 1 connection at a time we need to ignore the
   listening socket if we have a connection. If we add the listening socket 
   (socks0) to the fdsetR array then when a new connection comes in select
   will indicate that and will continue to do so until we accept it, so
   select will never timeout and we will never get our fading echos. 

   another approach would be to close the listening socket and then reopen it
   when the connection closes */

FD_ZERO (&fdsetR);
if (socks1 == 0)
   {
   FD_SET (socks0, &fdsetR);
   nfds = socks0 + 1;
   }
else
   {
   FD_SET (socks1, &fdsetR);
   nfds = socks1 + 1;
   }

/* Do the select. We are only interested in reading and a timeout so the
   write and exception arguments are null */

socksSelected = select (nfds, &fdsetR, (fd_set *) 0, (fd_set *) 0, &timeout);

if (socksSelected < 0)
   {
   perror ("tasking_echo: select returned with an error, give up at this point");
   exit (errno);
   }

/* if select returns a value greater than 0 it represents the number of
   sockets ready (for reading in this case).  */

if (socksSelected > 0)
   {
   if (FD_ISSET (socks0, &fdsetR))        /* the listening socket has */
      {                                   /* special handling */
      if (socks1 == 0)                    /* only do the accept if we are */
         {                                /* not already handling a 
                                             connection */
         clilen = sizeof (cli_addr);      /* this could be outside the loop */
                                          /* but I use it here so here it is */
         socks1 = accept (socks0, (struct sockaddr *) &cli_addr, &clilen);
         if (socks1 < 0)                  /* if accept returns < 0 we have */
            perror ("tasking_echo: accept error"); /* an error */
         else
            {
            peer_x = inet_ntoa (cli_addr.sin_addr);  /* get peers address */
            strcpy (&peer [0], peer_x);              /* and report it */

/* create an output file in the directory of the PM named echo-taskID_peerIP,peerPORT */
            sprintf (msg, "%secho-%d_%s,%d", output_path, myTaskID, peer,  ntohs (cli_addr.sin_port));
            outFile = fopen (msg, "w+");
            if (outFile == NULL)
               {
               savedErrno = errno;
               outFile = fopen ("%phx_cac_j14#m14_d02>CAC>Noah_Davids>stratus_world_2004>stcp_inetd_echo.error", "w+");
               if (outFile == NULL)
                  exit (savedErrno);
               fprintf (outFile, "Could not create output file %s\n", msg);
               fprintf (outFile, "errno = %d\n", savedErrno);
               exit (savedErrno);
               }

            fprintf (outFile, "tasking_echo: Connection from %s\n", &peer [0]);
      
            strcpy (&echostring [0], "");            /* initialize string */
            echostringlen = 0;                       /* and length */
      
/* non-bocking IO is not the same as TCP_NODELAY, it also is not inherited
   from listening socket so it must be done for every accepted socket */
   
            fcntl_flags = fcntl(socks1, F_GETFL, 0);
            if (fcntl_flags < 0)
               {
               perror ("tasking_echo: could not get the current fcntl flags for accepted socket");
               exit (errno);
               }
            fcntl_flags = fcntl_flags | O_NDELAY;
            if (fcntl(socks1, F_SETFL, fcntl_flags) < 0)
               {
               perror ("tasking_echo: could not set non blocking IO on accepted socket");
               exit (errno);
               }

            if (close (socks0) < 0)
               {
               perror ("tasking_echo: Could not close listening socket after accept");
               exit (errno);
               }
            open_listening_socket (&socks0, ka_timer, lingertime);
            } /* if (socks1 < 0 -- else */
         } /* if (socks1 == 0) */
      } /* if (FD_ISSET (socks0, &fdsetR) */

   if (FD_ISSET (socks1, &fdsetR))   /* if this socket is ready for */
      {                                /* reading */
      if (echostringlen > 2)         /* "remove" ending \n\r that */
          echostringlen = echostringlen - 2; /* were added */

/* receive bytes based on the number of bytes left in buffer. There will
   be bytes left in the buffer if we have not completed echoing the last
   string sent to us. We also need to leave room for the \n\r\0 that will be
   added to the end of the buffer. If there are more bytes in the receive
   queue then can fit in the buffer the next time select is called it will
   immediately indicate that the socket is ready for reading */

      recvBytes = recv (socks1, &echostring[echostringlen], BUFFERLEN - echostringlen - 3, 0);
      if (recvBytes == -1)  /* error during recv, clean up the socks */
         {                  /* and related arrays */
         sprintf (msg, "tasking_echo: Unexpected error while receiving on socket connected to %s", &peer[0]);
         perror (msg);
         socks1 = 0;
         peer [0] = 0x0;
         echostring [0] = 0x0;
         echostringlen = 0;
         fclose (outFile);
         }
      else                 /* no error on recv */
         {
         if (recvBytes == 0)  /* indicates connection was closed by peer */
            {
            fprintf (outFile, "tasking_echo: closing connection for %s\n", &peer[0]);
            if (close (socks1) < 0)  /* close our socket */
               {
               sprintf (msg, "tasking_echo: error closing socket connected to %s", &peer [0]);
               perror (msg);
               }
            socks1 = 0;     /* and clean up arrays */
            peer [0] = 0x0;
            echostring [0] = 0x0;
            echostringlen = 0;
            fclose (outFile);
            }
         else       /* we actually had bytes to read and we read them */
            {

/* just the number of bytes we have in the buffer to include what was just 
   read. Then add a \n\r\0 to it. The \n\r is needed so that when we echo
   the bytes we move the cursor down to the next line. Some clients, like
   telnet will not do that for the user. Without the \n\r the output becomes
   very strange. The \0 is needed so that %s in printf works correctly */

            oldechostringlen = echostringlen;
            echostringlen = echostringlen + recvBytes;
            echostring [echostringlen] = '\n';
            echostring [echostringlen+1] = '\r';
            echostring [echostringlen+2] = 0x0;
            echostringlen = echostringlen + 2;

/* now print the string but start from what was the end of the buffer so we 
   print only the bytes that were just received. */

            fprintf (outFile, "tasking_echo: received from %s: %d bytes %s\n", &peer [0], recvBytes, &echostring [oldechostringlen]);

/* now send the entire buffer, any old bytes and the new bytes */

            sendBytes = send (socks1, &echostring [0], echostringlen, 0);

/* if we didn't send all the bytes we had some kind of problem. Ideally at this
   point I could loop and try to send again or take some kind of exception
   problem. It is not strictly speaking an error. I choose just to report it.
   ALWAYS CHECK and take appropriate action, never assume that all characters
   were sent */

            if (sendBytes != echostringlen)
               fprintf (outFile, "tasking_echo: problem sending echo string to %s only %d out of %d characters sent\n", &peer [0], sendBytes, echostringlen);
            else

/* If all characters were sent OK echo that and then cut the buffer in half
   and remeber the last half for further echoing */

               fprintf (outFile, "tasking_echo: echoing to %s: %d bytes %s", &peer [0], echostringlen, &echostring [0]);
            if (echostringlen > 3) /* 2 of these are the \n and \r */
               {
               x = echostringlen / 2 + (echostringlen % 2);
               y = echostringlen - x + 1;
               strncpy (msg, &echostring [x - 1], y);
               strncpy (&echostring [0], msg, y);
               echostring [y] = 0x0;
               echostringlen = y;
               }
            else                        /* not enough left to remember */
               echostringlen = 0;       /* so just zero the length */
            } /* if (recvBytes == 0) -- else */
         } /* if (recvBytes == -1 -- else */
      } /* if (FD_ISSET (socks1, &fdsetR) */
   } /* if (socksSelected > 0) */
else

/* nothing was ready for reading so we have a timeout */
   {
   if (socksSelected == 0)
      {                /* if all we have is the listening socket we can */
      if (socks1 != 0) /* skip all this */
         {

/* this is pretty much the same code as above. The difference is in the
   the strings output to the terminal. Following the program name are 2
   colon caracters instead of 1. This is not a typo. I make it a rule
   never to output the same message from more than 1 spot. that way it is
   always obvious where the message came from */

         if (echostringlen > 0)   
            {
            sendBytes = send (socks1, &echostring [0], echostringlen, 0);
            if (sendBytes != echostringlen)
               fprintf (outFile, "tasking_echo:: problem sending echo string to %s only %d out of %d characters sent\n", &peer [0], sendBytes, echostringlen);
            else
               fprintf (outFile, "tasking_echo:: echoing to %s: %d bytes %s", &peer [0], echostringlen, &echostring [0]);
             if (echostringlen > 3)     /* 2 of these are the \n and \r */
               {
               x = echostringlen / 2 + (echostringlen % 2);
               y = echostringlen - x + 1;
               strncpy (msg, &echostring [x - 1], y);
               strncpy (&echostring [0], msg, y);
               echostring [y] = 0x0;
               echostringlen = y;
               }
            else
               echostringlen = 0;
            }
         } /* if (socks1 == 0) */
      } /* socksSelected == 0 */
   else /* socksSelected must be < 0 some kind of error from select */
      perror ("tasking_echo: Unexpected error from select");
   } /* if (socksSelected > 0) -- else */

goto again;
}

